SET SERVEROUTPUT ON
SET LINESIZE 2000
SET PAGESIZE 60
SPOOL PPSN_2_migration.log

SELECT 'Migration Started: ' || CURRENT_TIMESTAMP FROM DUAL;

SET ECHO ON

REM ###########################################################################
REM #  Description: Create PPS-N 2.0 database objects
REM #
REM #
REM #   Usage:
REM #   Input Parameters:
REM #   1) None
REM #   Output:
REM #   1) Terminal
REM #   Internal Script Parameters:
REM #   1) None
REM #
REM ###########################################################################
REM # Change History
REM #
REM # DATE       User Name         DESCRIPTION
REM # ---------- ----------------- -------------------------------------------
REM # 04/29/2014 Duncan Shelley    Created script
REM #
REM ##########################################################################

DECLARE
   -- CONSTANTS

   cc_schema_name   CONSTANT VARCHAR2 (35) := 'PPSNEPL';

   -- VARIABLES

   vc_sql_statement          VARCHAR2 (4000);
   vc_errm                   VARCHAR2 (64);
   vn_code                   NUMBER;
   vn_continue               NUMBER;

   vn_row_count              NUMBER;

   -----------------------------------------------------------------------------
   -- NAME:         check_for_ppsn2
   -- TYPE:         Function
   -- DESCRIPTSION: Check to see if PPS_N 2.0 object already created
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
   FUNCTION check_for_ppsn2
      RETURN NUMBER
   IS
   BEGIN
      SELECT COUNT (*)
        INTO vn_row_count
        FROM sys.dba_tables
       WHERE     owner = cc_schema_name
             AND table_name IN ('EPL_NDF_UPDATE_FILE'
                               ,'EPL_STATUS'
                               ,'EPL_IEN_GENERATOR'
                               ,'EPL_NDF_OUTGOING_DIFFERENCES'
                               ,'EPL_VISTA_UPDATE'
                               ,'EPL_VISTA_STATUS'
                               ,'TEMP_EPL_NDCS'
                               ,'EPL_PACKAGE_SIZES'
                               ,'EPL_DDI_SEVERITY'
                               ,'EPL_REMATCH_SUGGESTION'
                               ,'EPL_MATCH_TYPE');

      IF vn_row_count > 0
      THEN
         RETURN (1);
      END IF;

      SELECT COUNT (*)
        INTO vn_row_count
        FROM sys.dba_views
       WHERE     owner = cc_schema_name
             AND view_name IN ('FDB_GCNSEQNO_PEM_V'
                              ,'FDB_GCNSEQNO_PLBLW_V'
                              ,'FDB_MONOGRAPH_PEM_V'
                              ,'FDB_PLBLWARNINGS_V');

      IF vn_row_count > 0
      THEN
         RETURN (1);
      END IF;

      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         DBMS_OUTPUT.put_line (
            'check_for_ppsn2: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
         RETURN (1);
   END check_for_ppsn2;

   -----------------------------------------------------------------------------
   -- NAME:         create_ppsn_tables
   -- TYPE:         Function
   -- DESCRIPTSION: Create PPS-N 2.0 tables
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
   FUNCTION create_ppsn_tables
      RETURN NUMBER
   IS
   BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  create_ppsn_tables');

      -- Create sequences section -------------------------------------------------
      vc_sql_statement :=
         'CREATE SEQUENCE ppsnepl.update_file_sequence START WITH 1 INCREMENT BY 1 NOCACHE NOCYCLE';

      EXECUTE IMMEDIATE vc_sql_statement;

      -- Create tables section -------------------------------------------------

      -- Table PPSNEPL.EPL_NDF_OUTGOING_DIFFERENCES

      vc_sql_statement :=
            'CREATE TABLE ppsnepl.epl_ndf_outgoing_differences ('
         || ' ndf_outgoing_differences_id   NUMBER (38, 0) NOT NULL '
         || ',new_value                     VARCHAR2 (256)     NULL '
         || ',old_value                     VARCHAR2 (256)     NULL '
         || ',vista_file_number             VARCHAR2 (20) NOT NULL '
         || ',vista_field_number            VARCHAR2 (10) NOT NULL '
         || ',vista_ien                     VARCHAR2 (50) NOT NULL '
         || ',ndc_update_file_fk            NUMBER (38, 0) '
         || ',action_type                   VARCHAR2 (10) NOT NULL '
         || ',created_by                    VARCHAR2 (50) NOT NULL '
         || ',created_dtm                   TIMESTAMP (6) NOT NULL) '
         || 'TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE UNIQUE INDEX ppsnepl.pkndfoutgoingdifferences ON ppsnepl.epl_ndf_outgoing_differences (ndf_outgoing_differences_id) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      -- New constraint
      vc_sql_statement :=
         'ALTER TABLE PPSNEPL.epl_ndf_outgoing_differences ADD CONSTRAINT pkndfoutgoingdifferences PRIMARY KEY (ndf_outgoing_differences_id) USING INDEX ppsnepl.pkndfoutgoingdifferences';

      EXECUTE IMMEDIATE vc_sql_statement;


      -- Table PPSNEPL.EPL_IEN_GENERATOR

      vc_sql_statement :=
            'CREATE TABLE ppsnepl.epl_ien_generator ( '
         || 'CONCEPT             VARCHAR2(50) NOT NULL, '
         || 'LAST_USED_IEN       NUMBER(30 , 0) NOT NULL, '
         || 'CREATED_BY          VARCHAR2(50) NOT NULL, '
         || 'CREATED_DTM         TIMESTAMP NOT NULL, '
         || 'LAST_MODIFIED_BY    VARCHAR2(50), '
         || 'LAST_MODIFIED_DTM   TIMESTAMP) '
         || 'TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE INDEX PPSNEPL.EPL_IEN_PK_IDX ON PPSNEPL.EPL_IEN_GENERATOR (CONCEPT ASC) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      -- New constraint
      vc_sql_statement :=
         'ALTER TABLE PPSNEPL.EPL_IEN_GENERATOR ADD CONSTRAINT EPL_IEN_PK PRIMARY KEY (CONCEPT) USING INDEX PPSNEPL.EPL_IEN_PK_IDX';

      EXECUTE IMMEDIATE vc_sql_statement;


      -- Table PPSNEPL.EPL_NDF_UPDATE_FILE

      vc_sql_statement :=
            'CREATE TABLE ppsnepl.epl_ndf_update_file( '
         || ' ndf_update_file_id       NUMERIC (30) NOT NULL '
         || ',file_name                VARCHAR (200) '
         || ',comments                 VARCHAR (2000) '
         || ',status_id_fk             NUMERIC (30) NOT NULL '
         || ',prod_transmission_date   TIMESTAMP '
         || ',test_transmission_date   TIMESTAMP '
         || ',status_modified_date     TIMESTAMP '
         || ',file_directory_path      VARCHAR (500) '
         || ',created_by               VARCHAR (50) NOT NULL '
         || ',created_dtm              TIMESTAMP NOT NULL '
         || ',last_modified_by         VARCHAR (50) '
         || ',last_modified_dtm        TIMESTAMP) '
         || 'TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE UNIQUE INDEX ppsnepl.pkndfupdatefile ON ppsnepl.epl_ndf_update_file (ndf_update_file_id) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE INDEX ppsnepl.idxstatus_id ON ppsnepl.epl_ndf_update_file (status_id_fk) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_ndf_update_file ADD CONSTRAINT pkndfupdatefile PRIMARY KEY (ndf_update_file_id) USING INDEX ppsnepl.pkndfupdatefile';

      EXECUTE IMMEDIATE vc_sql_statement;


      -- Table PPSNEPL.EPL_STATUS

      vc_sql_statement :=
            'CREATE TABLE ppsnepl.epl_status( '
         || ' status_id           NUMERIC (30) NOT NULL '
         || ',status_name         VARCHAR (200) '
         || ',status_desc         VARCHAR (1000) '
         || ',created_by          VARCHAR (50) NOT NULL '
         || ',created_dtm         TIMESTAMP NOT NULL '
         || ',last_modified_by    VARCHAR (50) '
         || ',last_modified_dtm   TIMESTAMP) '
         || ' TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE UNIQUE INDEX ppsnepl.pkstatus ON ppsnepl.epl_status (status_id) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_status ADD CONSTRAINT pkstatus PRIMARY KEY (status_id) USING INDEX ppsnepl.pkstatus';

      EXECUTE IMMEDIATE vc_sql_statement;


      -- Table PPSNEPL.EPL_VISTA_UPDATE

      vc_sql_statement :=
            'CREATE TABLE ppsnepl.epl_vista_update( '
         || ' vista_update_id     NUMERIC (30) NOT NULL '
         || ',file_fk             NUMERIC (30) NOT NULL '
         || ',site                NUMERIC (30) NOT NULL '
         || ',comments            VARCHAR (2000) '
         || ',vista_status_id_fk  NUMERIC (30) NOT NULL '
         || ',transmission_date   TIMESTAMP) '
         || 'TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE UNIQUE INDEX ppsnepl.pkvistaupdate ON ppsnepl.epl_vista_update (vista_update_id) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE INDEX ppsnepl.idxvistastatusid ON ppsnepl.epl_vista_update (vista_status_id_fk) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_vista_update ADD CONSTRAINT pkvistaupdate PRIMARY KEY (vista_update_id) USING INDEX ppsnepl.pkvistaupdate';

      EXECUTE IMMEDIATE vc_sql_statement;


      -- Table PPSNEPL.EPL_VISTA_STATUS

      vc_sql_statement :=
            'CREATE TABLE ppsnepl.epl_vista_status( '
         || ' vista_status_id     NUMERIC (30) NOT NULL '
         || ',status_name         VARCHAR (200) '
         || ',status_desc         VARCHAR (1000) '
         || ',created_by          VARCHAR (50) NOT NULL '
         || ',created_dtm         TIMESTAMP NOT NULL '
         || ',last_modified_by    VARCHAR (50) '
         || ',last_modified_dtm   TIMESTAMP) '
         || 'TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE UNIQUE INDEX ppsnepl.pkvistastatus ON ppsnepl.epl_vista_status (vista_status_id) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_vista_status ADD CONSTRAINT pkvistastatus PRIMARY KEY (vista_status_id) USING INDEX ppsnepl.pkvistastatus';

      EXECUTE IMMEDIATE vc_sql_statement;


      -- Table PPSNEPL.EPL_DDI_SEVERITY

      vc_sql_statement :=
            'CREATE TABLE ppsnepl.epl_ddi_severity( '
         || ' epl_id              NUMBER (30, 0) NOT NULL '
         || ' ,name                VARCHAR2 (50) NOT NULL '
         || ' ,created_by          VARCHAR2 (50) NOT NULL '
         || ' ,created_dtm         TIMESTAMP (6) NOT NULL '
         || ' ,last_modified_by    VARCHAR2 (50) '
         || ' ,last_modified_dtm   TIMESTAMP (6)) '
         || ' TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE UNIQUE INDEX ppsnepl.epl_ddi_severity_pk ON ppsnepl.epl_ddi_severity (epl_id) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_ddi_severity ADD CONSTRAINT epl_ddi_severity_pk PRIMARY KEY (epl_id) USING INDEX ppsnepl.epl_ddi_severity_pk';

      EXECUTE IMMEDIATE vc_sql_statement;


      -- Table PPSNEPL.EPL_DRUG_DRUG_INTERACTION

      vc_sql_statement :=
            'CREATE TABLE ppsnepl.epl_drug_drug_interaction( '
         || ' epl_drug_drug_interaction_id   NUMBER (38) NOT NULL '
         || ' ,interaction_name               VARCHAR2 (200) NOT NULL '
         || '   ,drug_ingredient_1              NUMBER (30, 0) NOT NULL '
         || '   ,drug_ingredient_2              NUMBER (30, 0) NOT NULL '
         || '   ,severity                       NUMBER (30, 0) NOT NULL '
         || '   ,nationally_entered             NUMBER (1, 0) '
         || '   ,total_indexes                  NUMBER (38, 0) '
         || '   ,locally_edited                 NUMBER (1, 0) '
         || '   ,vista_ien                      NUMBER(30 , 0) NOT NULL '
         || '   ,created_by                     VARCHAR2 (50) NOT NULL '
         || '   ,created_dtm                    TIMESTAMP (6) NOT NULL '
         || '   ,inactivation_date              TIMESTAMP (6) '
         || '   ,last_modified_by               VARCHAR2 (50) '
         || '   ,last_modified_dtm              TIMESTAMP (6)) '
         || ' TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_drug_drug_interaction ADD CONSTRAINT uk_epl_ddi_1 UNIQUE (drug_ingredient_1, drug_ingredient_2, severity)';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE UNIQUE INDEX ppsnepl.epldrugdruginteractionid_pk ON ppsnepl.epl_drug_drug_interaction (epl_drug_drug_interaction_id) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE INDEX ppsnepl.drug_ingredient_1_idx ON ppsnepl.epl_drug_drug_interaction (drug_ingredient_1) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE INDEX ppsnepl.drug_ingredient_2_idx ON ppsnepl.epl_drug_drug_interaction (drug_ingredient_2) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE INDEX ppsnepl.severity_idx ON ppsnepl.epl_drug_drug_interaction (severity) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_drug_drug_interaction ADD CONSTRAINT epldrugdruginteractionid_pk  PRIMARY KEY (epl_drug_drug_interaction_id) USING INDEX ppsnepl.epldrugdruginteractionid_pk';

      EXECUTE IMMEDIATE vc_sql_statement;

      -- Table PPSNEPL.EPL_REMATCH_SUGGESTION

      vc_sql_statement :=
         'CREATE TABLE ppsnepl.epl_rematch_suggestion(
   epl_rematch_suggestion_id       NUMBER (38, 0) NOT NULL
  ,old_ien                         VARCHAR2 (50) NOT NULL
  ,new_ien                         VARCHAR2 (50) 
  ,match_type_id_fk                NUMBER (38, 0) NOT NULL
  ,ndc_update_file_fk              NUMBER (30, 0)
  ,match_scope                    VARCHAR2 (50)
  ,created_by                     VARCHAR (50) NOT NULL
  ,created_dtm                    TIMESTAMP NOT NULL
  ,last_modified_by               VARCHAR (50)
  ,last_modified_dtm              TIMESTAMP) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE UNIQUE INDEX ppsnepl.pkeplrematchsuggestion ON ppsnepl.epl_rematch_suggestion (epl_rematch_suggestion_id) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE INDEX ppsnepl.idxmatch_type_id ON ppsnepl.epl_rematch_suggestion (match_type_id_fk) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE INDEX ppsnepl.idxndc_update_file ON ppsnepl.epl_rematch_suggestion (ndc_update_file_fk) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_rematch_suggestion ADD CONSTRAINT eplrematchsuggestion_pk PRIMARY KEY (epl_rematch_suggestion_id) USING INDEX ppsnepl.pkeplrematchsuggestion';

      EXECUTE IMMEDIATE vc_sql_statement;


      -- Table PPSNEPL.EPL_MATCH_TYPE

      vc_sql_statement :=
         'CREATE TABLE ppsnepl.epl_match_type(
   match_type_id           NUMBER (38, 0) NOT NULL
  ,match_name               VARCHAR (200)
  ,match_desc                VARCHAR (1000)
  ,created_by                VARCHAR (50) NOT NULL
  ,created_dtm               TIMESTAMP NOT NULL
  ,last_modified_by        VARCHAR (50)
  ,last_modified_dtm         TIMESTAMP) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE UNIQUE INDEX ppsnepl.pkmatch_type ON ppsnepl.epl_match_type (match_type_id ) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_match_type ADD CONSTRAINT eplmatchtype_pk PRIMARY KEY (match_type_id ) USING INDEX ppsnepl.pkmatch_type ENABLE VALIDATE';

      EXECUTE IMMEDIATE vc_sql_statement;


      -- Create synonyms section -------------------------------------------------


      -- Create relationships section -------------------------------------------------

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_ndf_outgoing_differences ADD (CONSTRAINT fk_pkndfoutgoingfile FOREIGN KEY (ndc_update_file_fk) REFERENCES ppsnepl.epl_ndf_update_file (ndf_update_file_id))';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_ndf_update_file ADD (CONSTRAINT fk_ndfupdatefile_status FOREIGN KEY (status_id_fk) REFERENCES ppsnepl.epl_status (status_id))';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_vista_update ADD (CONSTRAINT fk_vistaupdate_vistastatus FOREIGN KEY (vista_status_id_fk) REFERENCES ppsnepl.epl_vista_status (vista_status_id))';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_drug_drug_interaction ADD CONSTRAINT fk_epl_ddi_epl_ingredients_1 FOREIGN KEY (drug_ingredient_1) REFERENCES ppsnepl.epl_ingredients (epl_id)';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_drug_drug_interaction ADD CONSTRAINT fk_epl_ddi_epl_ingredients_2 FOREIGN KEY (drug_ingredient_2) REFERENCES ppsnepl.epl_ingredients (epl_id)';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_drug_drug_interaction ADD CONSTRAINT fk_epl_ddi_epl_severity FOREIGN KEY (severity) REFERENCES ppsnepl.epl_ddi_severity (epl_id)';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_rematch_suggestion ADD CONSTRAINT fk_match_type FOREIGN KEY (match_type_id_fk) REFERENCES ppsnepl.epl_match_type (match_type_id)';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_rematch_suggestion ADD CONSTRAINT fk_ndc_update_file FOREIGN KEY (ndc_update_file_fk) REFERENCES ppsnepl.epl_ndf_update_file(ndf_update_file_id)';

      EXECUTE IMMEDIATE vc_sql_statement;


      -- Grant permissions section -------------------------------------------------

      vc_sql_statement :=
         'GRANT SELECT ON ppsnepl.epl_ndf_outgoing_differences TO PPSNEPL_READ_ONLY_ROLE';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'GRANT SELECT ON ppsnepl.epl_ien_generator TO PPSNEPL_READ_ONLY_ROLE';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'GRANT SELECT ON ppsnepl.epl_ndf_update_file TO PPSNEPL_READ_ONLY_ROLE';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'GRANT SELECT ON ppsnepl.epl_status TO PPSNEPL_READ_ONLY_ROLE';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'GRANT SELECT ON ppsnepl.epl_vista_update TO PPSNEPL_READ_ONLY_ROLE';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'GRANT SELECT ON ppsnepl.epl_vista_status TO PPSNEPL_READ_ONLY_ROLE';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'GRANT SELECT ON ppsnepl.epl_ddi_severity TO PPSNEPL_READ_ONLY_ROLE';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'GRANT SELECT ON ppsnepl.epl_rematch_suggestion TO PPSNEPL_READ_ONLY_ROLE';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'GRANT SELECT ON ppsnepl.epl_match_type TO PPSNEPL_READ_ONLY_ROLE';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'GRANT SELECT ON ppsnepl.epl_drug_drug_interaction TO PPSNEPL_READ_ONLY_ROLE';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'GRANT SELECT ON ppsnepl.update_file_sequence TO PPSNEPL_READ_ONLY_ROLE';

      EXECUTE IMMEDIATE vc_sql_statement;


      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_ppsn_tables');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_ppsn_tables: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_ppsn_tables: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'create_ppsn_tables: ' || vc_sql_statement);
         RETURN (1);
   END create_ppsn_tables;

   -----------------------------------------------------------------------------
   -- NAME:         create_ppsn_views
   -- TYPE:         Function
   -- DESCRIPTSION: Create PPS-N 2.0 views
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
   FUNCTION create_ppsn_views
      RETURN NUMBER
   IS
   BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  create_ppsn_views');

      -- Create views section -------------------------------------------------

      -- View PPSNEPL.FDB_GCNSEQNO_PEM_V

      vc_sql_statement :=
         'GRANT SELECT ON fdb_dif.fdb_gcnseqno_pem TO ppsnepl WITH GRANT OPTION';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE VIEW ppsnepl.fdb_gcnseqno_pem_v AS SELECT * FROM fdb_dif.fdb_gcnseqno_pem';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'GRANT SELECT ON ppsnepl.fdb_gcnseqno_pem_v TO ppsnepl_read_only_role';

      EXECUTE IMMEDIATE vc_sql_statement;


      -- View FDB_GCNSEQNO_PLBLW_V

      vc_sql_statement :=
         'GRANT SELECT ON fdb_dif.fdb_gcnseqno_plblw TO ppsnepl WITH GRANT OPTION';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE VIEW ppsnepl.fdb_gcnseqno_plblw_v AS SELECT * FROM fdb_dif.fdb_gcnseqno_plblw';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'GRANT SELECT ON ppsnepl.fdb_gcnseqno_plblw_v TO ppsnepl_read_only_role';

      EXECUTE IMMEDIATE vc_sql_statement;


      -- View PPSNEPL.FDB_MONOGRAPH_PEM_V

      vc_sql_statement :=
         'GRANT SELECT ON fdb_dif.fdb_monograph_pem TO ppsnepl WITH GRANT OPTION';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE VIEW ppsnepl.fdb_monograph_pem_v AS SELECT * FROM fdb_dif.fdb_monograph_pem';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'GRANT SELECT ON ppsnepl.fdb_monograph_pem_v TO ppsnepl_read_only_role';

      EXECUTE IMMEDIATE vc_sql_statement;


      -- View PPSNEPL.FDB_PLBLWARNINGS_V

      vc_sql_statement :=
         'GRANT SELECT ON fdb_dif.fdb_plblwarnings TO ppsnepl WITH GRANT OPTION';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE VIEW ppsnepl.fdb_plblwarnings_v AS SELECT * FROM fdb_dif.fdb_plblwarnings';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'GRANT SELECT ON ppsnepl.fdb_plblwarnings_v TO ppsnepl_read_only_role';

      EXECUTE IMMEDIATE vc_sql_statement;


      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  create_ppsn_views');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'create_ppsn_views: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'create_ppsn_views: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line ('create_ppsn_views: ' || vc_sql_statement);
         RETURN (1);
   END create_ppsn_views;

   -----------------------------------------------------------------------------
   -- NAME:         insert_ppsn2_data
   -- TYPE:         Function
   -- DESCRIPTSION: INSERT PPS-N 2.0 data into tables
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
   FUNCTION insert_ppsn2_data
      RETURN NUMBER
   IS
   BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  insert_ppsn2_data');

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_status (STATUS_ID,STATUS_NAME,STATUS_DESC,CREATED_BY, CREATED_DTM) '
         || 'VALUES (1,''Initiated'',''File Creation Process Has Been Initiated'',''Initial Load'',sysdate)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_status : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_status (STATUS_ID,STATUS_NAME,STATUS_DESC,CREATED_BY, CREATED_DTM) '
         || 'VALUES (2,''Created'',''All changes to drug information in PPS-N have been collected into a file and placed on the file system'',''Initial Load'',sysdate)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_status : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_status (STATUS_ID,STATUS_NAME,STATUS_DESC,CREATED_BY, CREATED_DTM) '
         || 'VALUES (3,''Transmitted to Test  FTP'',''Transmission of the Created file to the test FTP location has been completed'',''Initial Load'',sysdate)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_status : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_status (STATUS_ID,STATUS_NAME,STATUS_DESC,CREATED_BY, CREATED_DTM) '
         || 'VALUES (4,''Start of National VistA Processing'',''The Test VistA instance has started the update file process'',''Initial Load'',sysdate)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_status : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_status (STATUS_ID,STATUS_NAME,STATUS_DESC,CREATED_BY, CREATED_DTM) '
         || 'VALUES (5,''Completion of National VistA Processing'',''The Test VistA instance has processed the update file and is ready for testing. A user may now access the Update approval functionality of PPS-N'',''Initial Load'',sysdate)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_status : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_status (STATUS_ID,STATUS_NAME,STATUS_DESC,CREATED_BY, CREATED_DTM) '
         || 'VALUES (6,''Approved by Test'',''A user has marked this update as Approved in the update management area of PPS-N'',''Initial Load'',sysdate)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_status : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_status (STATUS_ID,STATUS_NAME,STATUS_DESC,CREATED_BY, CREATED_DTM) '
         || 'VALUES (7,''Rejected by Test'',''A user has marked this update as Rejected in the update management area of PPS-N'',''Initial Load'',sysdate)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_status : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_status (STATUS_ID,STATUS_NAME,STATUS_DESC,CREATED_BY, CREATED_DTM) '
         || 'VALUES (8,''Transmitted to Production VistA'',''Transmission of the approved file to the production FTP location has been completed'',''Initial Load'',sysdate)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_status : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_status (STATUS_ID,STATUS_NAME,STATUS_DESC,CREATED_BY, CREATED_DTM) '
         || 'VALUES (99,''NDF Update Process Error'',''An error occurred during NDF update file process'',''Initial Load'',sysdate)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_status : '
         || TO_CHAR (SQL%ROWCOUNT));

      COMMIT;

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_ndf_update_file (NDF_UPDATE_FILE_ID,FILE_NAME,COMMENTS,STATUS_ID_FK,PROD_TRANSMISSION_DATE,TEST_TRANSMISSION_DATE,STATUS_MODIFIED_DATE,FILE_DIRECTORY_PATH,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) '
         || ' VALUES (0,''PPS_0PRV_0NEW.DAT'',''Initial Seed Data'',8,SYSDATE,SYSDATE,SYSDATE,''PPS/PROD'',''Developer'',SYSDATE,''Developer'',SYSDATE)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_ndf_update_file : '
         || TO_CHAR (SQL%ROWCOUNT));


      COMMIT;

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_ien_generator (CONCEPT, LAST_USED_IEN, CREATED_BY, CREATED_DTM, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) '
         || 'VALUES (''EPL_DRUG_UNITS'', 10000, ''DEV'', SYSDATE, ''DEV'', SYSDATE)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_ien_generator : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_ien_generator (CONCEPT, LAST_USED_IEN, CREATED_BY, CREATED_DTM, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) '
         || 'VALUES (''EPL_DOSAGE_FORMS'', 10000, ''DEV'', SYSDATE, ''DEV'', SYSDATE)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_ien_generator : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_ien_generator (CONCEPT, LAST_USED_IEN, CREATED_BY, CREATED_DTM, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) '
         || 'VALUES (''EPL_INGREDIENTS'', 10000, ''DEV'', SYSDATE, ''DEV'', SYSDATE)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_ien_generator : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_ien_generator (CONCEPT, LAST_USED_IEN, CREATED_BY, CREATED_DTM, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) '
         || 'VALUES (''EPL_MANUFACTURERS'', 10000, ''DEV'', SYSDATE, ''DEV'', SYSDATE)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_ien_generator : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_ien_generator (CONCEPT, LAST_USED_IEN, CREATED_BY, CREATED_DTM, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) '
         || 'VALUES (''EPL_NDCS'', 10000, ''DEV'', SYSDATE, ''DEV'', SYSDATE)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_ien_generator : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_ien_generator (CONCEPT, LAST_USED_IEN, CREATED_BY, CREATED_DTM, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) '
         || 'VALUES (''EPL_PACKAGE_TYPES'', 10000, ''DEV'', SYSDATE, ''DEV'', SYSDATE)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_ien_generator : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_ien_generator (CONCEPT, LAST_USED_IEN, CREATED_BY, CREATED_DTM, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) '
         || 'VALUES (''EPL_PRODUCTS'', 10000, ''DEV'', SYSDATE, ''DEV'', SYSDATE)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_ien_generator : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_ien_generator (CONCEPT, LAST_USED_IEN, CREATED_BY, CREATED_DTM, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) '
         || 'VALUES (''EPL_VA_DISPENSE_UNITS'', 10000, ''DEV'', SYSDATE, ''DEV'', SYSDATE)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_ien_generator : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_ien_generator (CONCEPT, LAST_USED_IEN, CREATED_BY, CREATED_DTM, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) '
         || 'VALUES (''EPL_VA_DRUG_CLASSES'', 10000, ''DEV'', SYSDATE, ''DEV'', SYSDATE)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_ien_generator : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_ien_generator (CONCEPT, LAST_USED_IEN, CREATED_BY, CREATED_DTM, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) '
         || 'VALUES (''EPL_VA_GEN_NAMES'', 10000, ''DEV'', SYSDATE, ''DEV'', SYSDATE)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_ien_generator : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_ien_generator (CONCEPT, LAST_USED_IEN, CREATED_BY, CREATED_DTM, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) '
         || 'VALUES (''EPL_DRUG_DRUG_INTERACTION'', 10843, ''DEV'', SYSDATE, ''DEV'', SYSDATE)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_ien_generator : '
         || TO_CHAR (SQL%ROWCOUNT));

      COMMIT;

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_vista_status (VISTA_STATUS_ID, STATUS_NAME, STATUS_DESC, CREATED_BY, CREATED_DTM, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) '
         || 'VALUES (1, ''STARTED'', ''The process has star...'', ''system'', CURRENT_TIMESTAMP, ''system'', CURRENT_TIMESTAMP)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_vista_status : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_vista_status (VISTA_STATUS_ID, STATUS_NAME, STATUS_DESC, CREATED_BY, CREATED_DTM, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) '
         || 'VALUES (2, ''COMPLETED'', ''The process has comp...'', ''system'', CURRENT_TIMESTAMP, ''system'', CURRENT_TIMESTAMP)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_vista_status : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_ddi_severity (EPL_ID , NAME, CREATED_BY, CREATED_DTM, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) '
         || 'VALUES (1, ''CRITICAL'', ''360'', TIMESTAMP ''2014-05-09 13:02:39'', ''360'', TIMESTAMP ''2014-05-09 13:02:39'')';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_ddi_severity : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
            'INSERT INTO ppsnepl.epl_ddi_severity (EPL_ID , NAME, CREATED_BY, CREATED_DTM, LAST_MODIFIED_BY, LAST_MODIFIED_DTM) '
         || 'VALUES (2, ''SIGNIFICANT'', ''360'', TIMESTAMP ''2014-05-09 13:10:00'', ''360'', TIMESTAMP ''2014-05-09 13:10:00'')';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_ddi_severity : '
         || TO_CHAR (SQL%ROWCOUNT));

      COMMIT;

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  insert_ppsn2_data');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'insert_ppsn2_data: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'insert_ppsn2_data: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line ('insert_ppsn2_data: ' || vc_sql_statement);
         RETURN (1);
   END insert_ppsn2_data;

   -----------------------------------------------------------------------------
   -- NAME:         ndc_process_1
   -- TYPE:         Function
   -- DESCRIPTSION: Create epl_package_size table
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
   FUNCTION ndc_process_1
      RETURN NUMBER
   IS
   BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  ndc_process_1');

      -- Create temp table

      vc_sql_statement :=
         'CREATE TABLE ppsnepl.temp_epl_ndcs AS SELECT * FROM ppsnepl.epl_ndcs';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.temp_epl_ndcs ADD (CONSTRAINT temp_epl_ndcs_pk PRIMARY KEY (epl_id))';

      EXECUTE IMMEDIATE vc_sql_statement;

      -- Create epl_package_sizes table

      vc_sql_statement :=
         'CREATE TABLE ppsnepl.epl_package_sizes( 
 epl_id                 NUMBER (30, 0) NOT NULL ENABLE
,package_size           NUMBER (13, 6) NOT NULL ENABLE
,ndf_package_size_ien   NUMBER (30, 0)
,created_by             VARCHAR2 (50 BYTE) NOT NULL ENABLE
,created_dtm            TIMESTAMP (6) NOT NULL ENABLE
,last_modified_by       VARCHAR2 (50 BYTE)
,last_modified_dtm      TIMESTAMP (6))
TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE UNIQUE INDEX ppsnepl.epl_package_sizes_pki ON ppsnepl.epl_package_sizes (epl_id) TABLESPACE ppsnepl_data';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_package_sizes ADD (CONSTRAINT epl_package_sizes_pk PRIMARY KEY (epl_id) USING INDEX ppsnepl.epl_package_sizes_pki ENABLE VALIDATE)';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'CREATE SEQUENCE ppsnepl.package_size_sequence MINVALUE 1 MAXVALUE 9999999999999999999999999999 INCREMENT BY 1 START WITH 108 NOCACHE  NOORDER  NOCYCLE';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'GRANT SELECT ON ppsnepl.epl_package_sizes TO PPSNEPL_READ_ONLY_ROLE';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  ndc_process_1');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'ndc_process_1: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'ndc_process_1: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line ('ndc_process_1: ' || vc_sql_statement);
         RETURN (1);
   END ndc_process_1;

/* Per Jean Chen, removed and replaced by new process to load from VistA
   -----------------------------------------------------------------------------
   -- NAME:         ndc_process_2
   -- TYPE:         Function
   -- DESCRIPTSION: Insert epl_package_sizes data
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
   FUNCTION ndc_process_2
      RETURN NUMBER
   IS
   BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  ndc_process_2');

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,16778,5249,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,24000,2121,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2480,5286,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,7.7,5161,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,121,2083,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3700,5375,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,19000,5407,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,76,3641,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2.8,5174,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,6.82,5177,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2.55,5175,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,88.72,5170,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,22700,5250,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1.25,5173,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,188,5376,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3200,5408,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,101,5377,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,12.1,5221,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,57.7,5378,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.5,4574,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.025,4762,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.1,4763,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.6,4764,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.3,4765,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.25,4793,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.05,4830,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.09,4902,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.75,4904,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.8,4918,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';


      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.15,4943,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.65,4952,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.2,4970,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.972,4994,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.89,4996,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.85,5001,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.55,5002,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.4,5003,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.78,5016,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.42,5017,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.98,5022,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.51,5023,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.625,5039,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.72,5051,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.67,5126,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.015,5158,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.5,5190,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.15,5215,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.6,5231,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.015,5236,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.85,5242,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.98,5243,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.4,5278,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.42,5280,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.625,5281,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.3,5300,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.25,5302,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.09,5306,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.2,5312,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.78,5313,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.05,5316,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.75,5332,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.65,5335,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.72,5339,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.1,5357,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.89,5367,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.972,5368,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.67,5384,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.025,5391,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.51,5399,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.8,5433,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,0.55,5441,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,90,195,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,30,4289,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,30,162,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1,4471,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1,4470,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1,4469,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1,4318,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1,49,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2,50,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,118,4567,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,25000,85,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1000,7,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,500,34,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,10000,283,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,24,280,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,470,5222,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,75,2079,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,200,60,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,125,2085,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,100,4433,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,100,4432,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,100,6,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,80,835,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,15,470,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,59.14,4760,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3800,5251,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,120,193,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,60,194,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,56,909,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,28,48,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,10,4529,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,10,66,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,50,163,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,84,841,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,20,436,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,21,430,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,40,282,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,4,278,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,190,2092,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,110,936,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,170,5409,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,160,2088,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,380,5191,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,14,455,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,7,168,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,6,279,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,180,454,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,5000,35,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,5,626,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2.48,5152,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,135,2087,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,270,2097,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,540,2102,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,12,170,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,8,308,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,355,5192,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,237,5410,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,375,2100,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,300,522,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,150,1026,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,133,5287,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3,4263,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3,65,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2000,552,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,32,2073,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,240,1971,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,45,2075,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2500,45,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2.5,4670,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,396,5345,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,452,5252,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,113,5224,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,48,58,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,454,5379,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,480,1779,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,85,2080,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,410,5253,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,59,5223,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,296,5288,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,16,474,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,18,451,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,295,5411,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,25,187,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,250,59,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,360,633,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,770,5380,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,780,5413,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,148,5255,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,35,453,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,769,5254,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3840,5412,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,10.5,4666,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,946,5346,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,346,5414,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,400,490,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,710,5317,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,335,5193,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,59.1,5130,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,473.2,5131,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,236.6,4684,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,9,772,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,42,908,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,112,2082,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,177,5256,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,275,3968,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,36,693,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,887,5289,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,473,3645,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,476,5381,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,12000,2118,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,22,166,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,70,1080,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,118.28,4717,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1600,5416,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,474,5194,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,4000,2112,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3785,598,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,365,2000,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,320,5415,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,63,1393,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,198,5195,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,244,5196,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,89,5382,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,72,1620,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2270,5197,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,238,5347,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,354,5290,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,236,5383,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,144,1929,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,14.79,5125,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,600,2103,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,450,807,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,227,5198,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3.75,2070,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,118.25,5124,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3780,5322,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,590,5350,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,141,5258,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,20000,1394,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,225,2095,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,750,4288,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,71,5348,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,14.2,4925,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,96,535,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1.8,4949,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,12500,5199,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,7.5,4565,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,527,5320,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,28.4,4629,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,510,5293,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,284,5292,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1020,5417,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,6.5,5087,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,18925,5349,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,102,4510,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,102,4509,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,800,2105,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,220,5257,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,168,2090,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,81,5291,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,108,4207,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,130,717,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,453,5319,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,44,4571,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,44,4570,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,23,5318,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,113.3,5145,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1500,2108,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,88,5259,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,960,5418,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,5.5,4932,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1014,5385,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,13,3879,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,19,2072,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,14.7,4909,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,8.5,5133,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,6.7,4966,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,200.0839,5132,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1920,5225,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,174,5419,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,370,5323,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,355.2,5136,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,11,3637,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,236.5,4862,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,17,3638,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,226,5321,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,140,1426,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,56.7,4628,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1.5,4790,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,29,3881,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3000,1904,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,64,2078,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,27,1434,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,420,5386,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,184,5294,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,385,5387,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,176,5388,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,42.5,4664,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,720,2104,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,266,5351,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,700,3646,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,51,3640,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,16.5,4576,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,16.5,4575,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,91,5200,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,119,5295,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,10.15,5142,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3.5,3333,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,114,1476,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,65,5420,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,175,2091,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,124,2084,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,325,1004,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,20.3,5139,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,290,5226,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,900,5352,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,155,5227,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,350,2099,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,477,5202,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,245,5203,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,730,5204,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,178,5353,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,113.4,4620,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,219.7,4626,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,92,4573,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,28.35,4630,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,453.6,4631,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,57,5201,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,118.3,4634,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,120.1,4657,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,142,5389,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,230,5297,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,43,5205,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,53,5296,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1100,5324,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,14.4,4680,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2.4,4681,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,255,5298,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,283.4,4683,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,87,4376,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,14000,5206,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,198.6,4688,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,85.05,4690,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,198.45,4691,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,199,5354,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,28.3,4693,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3500,5260,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,207,5228,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,7000,5421,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2400,5229,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,236.56,4697,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,224,5261,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,330,5230,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,331.2,4715,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,18.77,4716,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,6.26,4718,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,20.04,4719,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,424.96,4720,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,81.2,4726,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,11.6,4727,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,17000,5422,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,5500,4446,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,27000,5299,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,177.4,4708,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2268,5262,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,35.4,4709,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,86,5263,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,226.8,4711,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,85.2,4723,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,430,937,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,9000,5207,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,210,2094,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1200,1389,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,280,4364,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1008,52,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,67.5,4748,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,50000,44,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,105,5390,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,95,2081,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,4.6,4754,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,850,5355,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,34.7,4744,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,980,5356,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,340,5264,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1400,4525,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,9.9,4766,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,106,5424,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,53.2,4796,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,414,5392,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,283,5301,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,283.5,4780,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,129,5208,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,37.5,4782,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,121.9,4783,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,122,5423,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1280,4317,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,340.2,4773,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,170.3,4774,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,170.1,4775,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,56.25,4797,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,14.1,4798,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,399,5209,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,798,5358,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,66,5265,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,45000,753,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,44.4,4808,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,7.4,4810,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,165,2089,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,19200,5266,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,44.3,4813,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3600,5359,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,55,688,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,14.8,4817,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,9.3,4818,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,9.8,4566,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,9.12,4819,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,7500,5325,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,4.4,4820,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2.2,4821,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1.3,4822,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,41.3,4824,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,15000,647,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,128,2086,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,368,5361,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,369,5232,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1368,5360,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1254,5393,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,861,5211,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,575,5365,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,822,5394,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,538,5268,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,570,5270,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,390,5269,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,397,5271,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,576,5327,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,539,5427,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,630,5426,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,398,5303,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,371,5272,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,285,5304,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,381,5364,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,947,5210,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,472,5212,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,213,5233,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,260,2096,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,660,5425,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,222,5234,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,425,5267,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1040,5362,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,662,5363,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,426,5326,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,862,5235,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1366,5273,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,33,3639,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,61,5330,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,54,1475,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,78,5213,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,38,5329,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,46,2076,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1.45,4866,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,37,5328,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,88.7,4878,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,144.56,4879,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,274,5429,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,39,5428,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2.7,4876,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,349,5305,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,126,4444,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,52,2077,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1.2,4891,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,181,5274,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,29.6,4892,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,103.5,4901,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,112.5,4894,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,212,5331,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,97.5,4896,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,16.8,4897,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,11350,5214,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,159,5430,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,226.79,4900,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,682,1237,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,406,5431,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,373,5432,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,432,4218,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,256,4298,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,288,2098,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,16.7,4908,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,8000,2024,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,4.5,4914,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,100.8,4915,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,95.1,4919,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,30.5,4917,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,620,4290,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,4350,5333,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,4.9,4923,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,16.9,4924,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,26,161,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,163,5307,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,115,3936,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,312,5434,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,562,5308,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,12.5,4934,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,99,3642,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1800,5237,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,34,2074,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,187.5,4941,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,29.57,4938,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,131,4577,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,141.7,4940,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,4500,3648,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,164,5311,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,99.2,4945,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3552,5334,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,47.1,4950,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,56.6,4499,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,532,5310,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,708,5238,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,49,5395,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1816,5309,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1814,5366,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1810,5435,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2720,5216,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1080,5436,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,120.49,4961,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,67,4429,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,66.5,4963,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,141.8,4964,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,70.5,4965,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,127,4568,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,35.44,4967,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,98,5437,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,12.9,4983,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,640,5336,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,195,5396,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1892,5275,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,117,4582,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,580,5239,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,345,5438,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,354.84,4977,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,473.12,4978,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,465,5217,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,132,5276,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,330.6,4985,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,197.2,4986,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,4.2,4987,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3.8,4988,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3.2,4989,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,189,5240,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,74,5337,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,56.8,4997,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,73,5439,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,6.9,4995,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,252,4398,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,58,5397,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,28000,5241,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,216,4316,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,205,5277,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,448,5440,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3178,5398,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,50.5,5004,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,19.8,2029,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,18000,2119,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,31,4440,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,31,4179,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,8500,2116,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,10.6,5018,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,88.71,5014,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1.17,5015,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1.6,5019,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,315,5279,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,52.5,5021,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1.74,5024,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,8.1,5025,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,40000,2124,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,504,5218,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,62.5,5030,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,948,5219,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,62.1,5032,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,993.6,5033,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,930,5442,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,228,4426,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,11500,5338,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,22500,5443,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,103,5444,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,12.34,5041,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,41,5369,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,6000,4496,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,6000,4495,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,6000,4494,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,6000,51,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1.7,2069,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,9.5,5052,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3.9,5045,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,79,5445,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,794,5446,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,107,5370,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,840,5340,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,455,5220,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1362,5282,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,681,5244,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,221,5401,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,107.7,5060,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,415,5400,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2.3,5053,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,21.25,5061,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,251.3,5062,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,251,5283,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,235,5245,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,22.2,5065,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,69,5341,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,9990,5372,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2800,5373,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,243,5371,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,478,5343,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,5600,5447,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,231,5342,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,6.6,5072,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3785.41,5078,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3.3,5073,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,6.1,5074,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,437,5402,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1419,5403,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,147,5247,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,378,5246,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,239.4,5076,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,11.3,5081,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,70.8,5082,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,310,4292,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,56.75,5085,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,62,4174,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,18.9,5086,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,956,5374,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,10.2,5091,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,8.6,5092,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,336,5344,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,5.8,5090,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,901,5284,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,3.7,5089,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,9.4,5103,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,5.6,5104,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,5.95,5105,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,82.5,5106,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,59.7,5107,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,11.9,5108,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,5.25,5109,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,23.6,5098,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,46.6,5099,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,23.3,5100,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,7.1,5110,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,440,3880,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,127.6,5095,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,192,5404,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,297,5248,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,156,5448,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,1900,5314,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,8.7,5112,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,7.3,5113,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,22.5,5120,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,14.17,5115,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,236.58,5116,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,118.29,5117,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,444,5285,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,621,5405,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2.6,4545,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2.6,4544,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2.6,4543,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,2.6,4542,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,35.5,5129,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,77.96,5128,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,4.1,5159,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,8.71,5160,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,68,5315,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,104,3643,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,225000,5406,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into PPSNEPL.EPL_PACKAGE_SIZES (EPL_ID,PACKAGE_SIZE,NDF_PACKAGE_SIZE_IEN,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (ppsnepl.package_size_sequence.nextval,125000,3649,''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''),''MIGRATOR'',to_timestamp(''21-MAY-14 08.05.05.451000000 PM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      COMMIT;

      EXECUTE IMMEDIATE 'SELECT COUNT (*) FROM ppsnepl.epl_package_sizes'
         INTO vn_row_count;


      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) inserted into ppsnepl.epl_package_sizes : '
         || TO_CHAR (vn_row_count));

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  ndc_process_2');

      IF vn_row_count = 0
      THEN
         RETURN (1);
      END IF;

      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'ndc_process_2: Error code ' || vn_code || ': ' || vc_errm);
         RETURN (1);
   END ndc_process_2;
*/

   -----------------------------------------------------------------------------
   -- NAME:         ndc_process_3
   -- TYPE:         Function
   -- DESCRIPTSION: Process ndcs and package size
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
   FUNCTION ndc_process_3
      RETURN NUMBER
   IS
   BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  ndc_process_3');

      vc_sql_statement :=
         'UPDATE ppsnepl.epl_ndcs SET package_size = NULL WHERE 1 = 1';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_ndcs MODIFY package_size NUMBER (30, 0)';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_ndcs RENAME COLUMN package_size TO package_size_id_fk';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'UPDATE ppsnepl.epl_ndcs o
   SET o.package_size_id_fk =
          (SELECT MIN (epl_package_sizes.epl_id)
             FROM ppsnepl.epl_package_sizes, ppsnepl.temp_epl_ndcs
            WHERE     epl_package_sizes.package_size =
                         temp_epl_ndcs.package_size
                  AND temp_epl_ndcs.epl_id = o.epl_id)
 WHERE o.epl_id IN (SELECT epl_id FROM ppsnepl.temp_epl_ndcs)';

      EXECUTE IMMEDIATE vc_sql_statement;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) updated in ppsnepl.epl_ndcs : '
         || TO_CHAR (SQL%ROWCOUNT));

      vc_sql_statement :=
         'CREATE INDEX ppsnepl.epl_ndc_pckg_size_fk_idx ON ppsnepl.epl_ndcs (package_size_id_fk)';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement :=
         'ALTER TABLE ppsnepl.epl_ndcs ADD FOREIGN KEY(package_size_id_fk) REFERENCES ppsnepl.epl_package_sizes (epl_id)';

      EXECUTE IMMEDIATE vc_sql_statement;

      vc_sql_statement := 'DROP SEQUENCE ppsnepl.package_size_sequence';

      EXECUTE IMMEDIATE vc_sql_statement;

      COMMIT;

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  ndc_process_3');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'ndc_process_3: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'ndc_process_3: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line ('ndc_process_3: ' || vc_sql_statement);
         RETURN (1);
   END ndc_process_3;

   -----------------------------------------------------------------------------
   -- NAME:         alter_ppsn2_tables
   -- TYPE:         Function
   -- DESCRIPTSION: Alter PPSN2 tables
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
   FUNCTION alter_ppsn2_tables
      RETURN NUMBER
   IS
   BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  alter_ppsn2_tables');

      vc_sql_statement := 'ALTER TABLE  ppsnepl.epl_products ADD (
        rxcui NUMBER(30 , 0) ,
        rx_norm_name VARCHAR2(600) ,
        rx_norm_syn VARCHAR2(600),
        rx_norm_tty VARCHAR2(50),
        rx_norm_language VARCHAR2(3),
        rx_norm_supress CHAR(1),
        rx_norm_umlscui VARCHAR2(50))';

      EXECUTE IMMEDIATE vc_sql_statement;

      COMMIT;

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  alter_ppsn2_tables');
      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'alter_ppsn2_tables: Error code ' || vn_code || ': ' || vc_errm);
         sys.DBMS_OUTPUT.put_line (
            'alter_ppsn2_tables: last value for vc_sql_statement ');
         sys.DBMS_OUTPUT.put_line (
            'alter_ppsn2_tables: ' || vc_sql_statement);
         RETURN (1);
   END alter_ppsn2_tables;

   -----------------------------------------------------------------------------
   -- NAME:
   -- TYPE:         Function
   -- DESCRIPTSION: Insert data into epl_national_settings table
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
   FUNCTION insert_epl_national_settings
      RETURN NUMBER
   IS
   BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  insert_epl_national_settings');

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) Values (35, ''NDF_UPDATE_RUN_STATE'', ''COMPLETED'', ''DEVELOPER'', SYSDATE, ''NdfUpdateProcess'', SYSDATE)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) Values   (36, ''NDF_UPDATE_LAST_RUN'', NULL, ''DEVELOPER'', TO_TIMESTAMP(''1/31/2014 12:00:00.000000 AM'',''fmMMfm/fmDDfm/YYYY fmHH12fm:MI:SS.FF AM''),''NdfUpdateProcess'', SYSDATE)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (37,''NDF_OUTGOING_FILE_TEXT'',''unmatched from the National Drug File (NDF).  The VA Product |name and CMOP ID corresponding to the unmatched local drug file |name are listed on the indented line beneath each entry.  An |Inactivation Date may be listed for entries when this reason |applies.  Until you rematch these entries to NDF, they will not |transmit to CMOP and drug-drug interaction checks will not check |for these products.  It is critical that you rematch these |products immediately.  You may also need to match a new |orderable item.  Any possible dosages and local possible |dosages for these unmatched products have been deleted. |Therefore, the dosages for each unmatched product should |be reviewed after the rematch or recreated if the product |can not be rematched to a VA Product through the NDF |'',null,null,null,null,''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''),''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (38,''NDF_OUTPUT_FILE_DIRECTORY'',''./tmp/'',null,null,null,null,''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''),''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (39,''NDF_OUTPUT_TST_PROTOCOL'',''FTP'',null,null,null,null,''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''),''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (40,''NDF_OUTPUT_TST_HOSTNAME'',''10.1.19.9'',null,null,null,null,''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''),''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (41,''NDF_OUTPUT_TST_PORT'',''21'',null,null,null,null,''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''),''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (42,''NDF_OUTPUT_TST_USERNAME'',''USRVISTA'',null,null,null,null,''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''),''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (43,''NDF_OUTPUT_TST_PASSWORD'','' '',null,null,null,null,''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''),''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (45,''NDF_OUTPUT_TST_DIRECTORY'',''$1$DGA55:[PMIS.MARK]'',null,null,null,null,''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''),''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (46,''NDF_OUTPUT_PROD_PROTOCOL'',''FTP'',null,null,null,null,''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''),''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (47,''NDF_OUTPUT_PROD_HOSTNAME'',''10.1.19.9'',null,null,null,null,''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''),''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (48,''NDF_OUTPUT_PROD_PORT'',''21'',null,null,null,null,''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''),''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (49,''NDF_OUTPUT_PROD_USERNAME'',''USRVISTA'',null,null,null,null,''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''),''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (50,''NDF_OUTPUT_PROD_PASSWORD'','' '',null,null,null,null,''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''),''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (51,''NDF_OUTPUT_PROD_DIRECTORY'',''$1$DGA55:[PMIS.MARK]'',null,null,null,null,''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''),''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (52,''NDF_OUTPUT_FTP_ON'',null,null,null,0,null,''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''),''DEVELOPER'',to_timestamp(''21-MAY-14 12.00.00.000000000 AM'',''DD-MON-RR HH.MI.SS.FF AM''))';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (53,''MESSAGE_STRENGTH_TEXT'','' The following Strength(s) have been edited or added. Any product matched to| these products will be unmatched. If site wants to continue to use the| product the site must rematch to local file #50 entries to the listed VA| products.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (54,''MESSAGE_OVERRIDEDOSECHECK_TEXT'','' The override dose form checks field has changed for the following products.|'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (55,''MESSAGE_NATIONALFORMULARY_TEXT'','' The National Formulary Indicator has changed for the following| VA Products.  The National Formulary Indicator will automatically| be changed in your local DRUG file (#50).  Please review the| VISN and Local Formulary designations of these products and| make appropriate changes.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (56,''MESSAGE_INACTIVATEDCMOP_TEXT'','' The following active VA Products are no longer marked for CMOP.| All local drug file entries matched to these VA Products will| be UNMARKED for CMOP.  In order to have these entries dispensed| by CMOP any local DRUG file (#50) entries matched to these| products must be re-matched to another VA product that is actively| marked for CMOP dispensing.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (57,''MESSAGE_ADDEDPRODUCT_TEXT'','' The following VA Products have been added to the National| Drug File.  You may wish to review, then match or unmatch| local drug file entries based on this updated information.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (58,''MESSAGE_SCHEDULECHANGE_TEXT'','' The CS Federal Schedule for the following VA Products has| been added or changed.  You may wish to review the DRUG| file (#50) and make appropriate changes to the DEA, SPECIAL| HDLG field (#3).'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (59,''MESSAGE_POSSIBLEDOSAGE_TEXT'','' The Auto-Create Possible Dosages settings have been edited for the| following VA Products. Please review your local dosages for products| matched to these entries. Edits to your site''''s possible dosages or| local possible dosages may be needed.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (60,''MESSAGE_PRINTNAMECHANGE_TEXT'','' The following VA Print Name(s) have been edited or added. Any product matched| to these products will be unmatched. If site wants to continue to use the| the product the site must rematch to local file #50 entries to the listed VA| product the site must rematch to local file #50 entries to the listed VA| product.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (61,''MESSAGE_DRUGUNITCHANGE_TEXT'','' The following Unit(s) have been edited or added. Any product matched to these| products will be unmatched. If site wants to continue to use the product the| site must rematch to local file #50 entries to the listed VA Product.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (62,''MESSAGE_DISPENSEUNITCHANGE_TEXT'','' The following VA Dispense Units(s) have been edited or added. Any product| matched to these products will be unmatched. If site wants to continue to use| the product the site must rematch to local file #50 entries to the listed VA| product.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (63,''MESSAGE_VACLASSCHANGE_TEXT'','' The following VA Drug Class(es) have been edited or added. The VA Class| for this product will be automatically updated in file #50.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (64,''MESSAGE_CMOPCHANGE_TEXT'','' The following VA Product Identifier(s) have been edited or added. Any product| matched to these products will be unmatched. If site wants to continue to use| the product the site must rematch to local file #50 entries to the listed VA| product.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (65,''MESSAGE_FUTUREINACTIVE_TEXT'','' The following products will be inactivated on the date listed. No alternative| products have been found.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (66,''MESSAGE_SCHEDULECHANGEALL_TEXT'','' The following CS Federal Schedule(s) have been edited or added. Please review| the local DEA special handling field and make edits are required.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (67,''MESSAGE_GENERICNAMECHANGE_TEXT'','' The following VA Generic Name(s) have been edited or added. Any product| matched to these products will be unmatched. If site wants to continue to use| the product the site must rematch to local file #50 entries to the listed VA| product.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (68,''MESSAGE_DOSAGEFORMCHANGE_TEXT'','' The following VA Dosage Form(s) have been edited. Any product matched to| these products will be unmatched. If site wants to continue to use the| product the site must rematch to local file #50 entries to the listed VA| product.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (69,''MESSAGE_INACTIVATEDPRODUCT_TEXT'','' The following product(s) have been inactivated. No alternative products have| been found.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (70,''MESSAGE_REACTIVATEDPRODUCT_TEXT'','' The following VA Product(s) have been reactivated.|'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (71,''MESSAGE2_MEDGUIDE_TEXT'','' The FDA Med Guide for the following products has been changed|'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (72,''MESSAGE2_EXCLUEDEDDDI_TEXT'','' The following products have been flagged for exclusion from drug-drug| interaction checks.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (73,''MESSAGE2_PREVEXCLUEDEDDDI_TEXT'','' The following products, previously flagged for exclusion from drug-drug| interaction checks, have been changed to be included| in drug-drug interaction checks where appropriate.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (74,''MESSAGE2_DDI_TEXT'','' The following interactions in National Drug File (NDF) have been added,| edited or inactivated.  These changes are the result of review and| recommendations from the NDF support group.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'INSERT INTO ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (75,''NATIONAL_SITE_NUMBER'',NULL,442,NULL,NULL,NULL,''DEVELOPER'',SYSDATE,''DEVELOPER'',SYSDATE)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (76,''MESSAGE_INACTIVATEDWSUGG_TEXT'',''The following product(s) have been inactivated. Any file #50 entries matched|to these products will be unmatched. To continue to use the product, the site|must rematch the local file #50 entries to the listed VA Product.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (77,''MESSAGE_PROPOSEDINACTIVATEDWSUGG_TEXT'',''The following product(s) WILL be inactivated in on the date listed. Any file|#50 entries matched to these products will be unmatched once the product is|inactivated. If a site wants to continue to use the product, the site must|rematch the local file #50 entries to the listed VA Product.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'INSERT INTO ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) VALUES (78, ''REPORT_DATA_UPDATE_START'', null, null, null, null, null, ''DEVELOPER'', sysdate, ''DEVELOPER'', sysdate)';

      EXECUTE IMMEDIATE
         'INSERT INTO ppsnepl.epl_national_settings (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) VALUES (79, ''REPORT_DATA_UPDATE_COMPLETE'', null, null, null, null, null, ''DEVELOPER'', sysdate, ''DEVELOPER'', sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.EPL_NATIONAL_SETTINGS (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (80,''MESSAGE_INACTIVATEDREMATCHED_TEXT'',''The following product(s) have been inactivated. Any file|#50 entries matched to these products will be unmatched and rematched to the|local file #50 entries listed.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.EPL_NATIONAL_SETTINGS (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (81,''MESSAGE_OTHERREMATCHED_TEXT'',''The following product(s) have been unmatched. Any file|#50 entries matched to these products will be unmatched and rematched to the|local file #50 entries listed.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.EPL_NATIONAL_SETTINGS (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (82,''MESSAGE_OTHERREMATCHSUGG_TEXT'',''The following product(s) have been unmatched Any file|#50 entries matched to these products will be unmatched. To continue to use|the product, the site must rematch the local file #50 entries to the listed VA Product.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      EXECUTE IMMEDIATE
         'Insert into ppsnepl.EPL_NATIONAL_SETTINGS (ID,KEY_NAME,STRING_VALUE,INTEGER_VALUE,DECIMAL_VALUE,BOOLEAN_VALUE,DATE_VALUE,CREATED_BY,CREATED_DTM,LAST_MODIFIED_BY,LAST_MODIFIED_DTM) values (83,''MESSAGE_OTHERNOREMATCH_TEXT'',''The following product(s) have been unmatched. No alternative products|have been found.'',null,null,null,null,''DEVELOPER'',sysdate,''DEVELOPER'',sysdate)';

      COMMIT;

      EXECUTE IMMEDIATE 'SELECT COUNT (*) FROM ppsnepl.epl_national_settings'
         INTO vn_row_count;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) in ppsnepl.epl_national_settings : '
         || TO_CHAR (vn_row_count));

      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   END:  insert_epl_national_settings');

      IF vn_row_count = 0
      THEN
         RETURN (1);
      END IF;

      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
               'insert_epl_national_settings: Error code '
            || vn_code
            || ': '
            || vc_errm);
         RETURN (1);
   END insert_epl_national_settings;

   -----------------------------------------------------------------------------
   -- NAME:         insert_match_type
   -- TYPE:         Function
   -- DESCRIPTSION: Insert data into epl_match_type table
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
   FUNCTION insert_match_type
      RETURN NUMBER
   IS
   BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line ('INFORMATION -   START:  insert_match_type');

      EXECUTE IMMEDIATE
         'insert into ppsnepl.epl_match_type (match_type_id, match_name, match_desc, created_by, created_dtm) values (1, ''INACTIVATE'', ''PPSN INACTIVATED PRODUCTS'',''DEVELOPER'', sysdate)';

      EXECUTE IMMEDIATE
         'insert into ppsnepl.epl_match_type (match_type_id, match_name, match_desc, created_by, created_dtm) values (2, ''FUTURE INACTIVATE'', ''PPSN FUTURE INACTIVATED PRODUCTS'',''DEVELOPER'', sysdate)';

      EXECUTE IMMEDIATE
         'insert into ppsnepl.epl_match_type (match_type_id, match_name, match_desc, created_by, created_dtm) values (3, ''STRENGTH CHANGE'', ''LOCAL UNMATCH DUE TO STRENGTH CHANGE'',''DEVELOPER'', sysdate)';

      EXECUTE IMMEDIATE
         'insert into ppsnepl.epl_match_type (match_type_id, match_name, match_desc, created_by, created_dtm) values (4, ''PRINT NAME CHANGE'', ''LOCAL UNMATCH DUE TO PRINT NAME CHANGE'',''DEVELOPER'', sysdate)';

      EXECUTE IMMEDIATE
         'insert into ppsnepl.epl_match_type (match_type_id, match_name, match_desc, created_by, created_dtm) values (5, ''DISPENSE UNIT CHANGE'', ''LOCAL UNMATCH DUE TO DISPENSE UNIT CHANGE'',''DEVELOPER'', sysdate)';

      EXECUTE IMMEDIATE
         'insert into ppsnepl.epl_match_type (match_type_id, match_name, match_desc, created_by, created_dtm) values (6, ''PRODUCT IDENTIFIER CHANGE'', ''LOCAL UNMATCH DUE TO PRODUCT IDENTIFIER CHANGE'',''DEVELOPER'', sysdate)';

      EXECUTE IMMEDIATE
         'insert into ppsnepl.epl_match_type (match_type_id, match_name, match_desc, created_by, created_dtm) values (7, ''GENERIC NAME CHANGE'', ''LOCAL UNMATCH DUE TO GENERIC NAME CHANGE'',''DEVELOPER'', sysdate)';

      EXECUTE IMMEDIATE
         'insert into ppsnepl.epl_match_type (match_type_id, match_name, match_desc, created_by, created_dtm) values (8, ''DOSAGE FORM CHANGE'', ''LOCAL UNMATCH DUE TO DOSAGE FORM CHANGE'',''DEVELOPER'', sysdate)';

      EXECUTE IMMEDIATE
         'insert into ppsnepl.epl_match_type (match_type_id, match_name, match_desc, created_by, created_dtm) values (9, ''UNITS CHANGE'', ''LOCAL UNMATCH DUE TO UNITS CHANGE'',''DEVELOPER'', sysdate)';

      EXECUTE IMMEDIATE
         'insert into ppsnepl.epl_match_type (match_type_id, match_name, match_desc, created_by, created_dtm) values (10, ''CS FED SCHEDULE CHANGE'', ''LOCAL UNMATCH DUE TO CS FEDERAL SCHEDULE CHANGE CHANGE'',''DEVELOPER'', sysdate)';

      COMMIT;

      EXECUTE IMMEDIATE 'SELECT COUNT (*) FROM ppsnepl.epl_match_type'
         INTO vn_row_count;

      sys.DBMS_OUTPUT.put_line (
            'INFORMATION -   Number of row(s) in ppsnepl.epl_match_type : '
         || TO_CHAR (vn_row_count));

      sys.DBMS_OUTPUT.put_line ('INFORMATION -   END:  insert_match_type');

      IF vn_row_count = 0
      THEN
         RETURN (1);
      END IF;

      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
            'insert_match_type: Error code ' || vn_code || ': ' || vc_errm);
         RETURN (1);
   END insert_match_type;

   -----------------------------------------------------------------------------
   -- NAME:         update_national_setting
   -- TYPE:         Function
   -- DESCRIPTSION: Update the epl_national_settings table
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
   FUNCTION update_national_setting
      RETURN NUMBER
   IS
   BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  update_national_setting');

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The following Strength(s) have been edited or added. Any product matched to|these products will be unmatched. If site wants to continue to use the|product the site must rematch to local file #50 entries to the listed VA|products.'' where id =53';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The override dose form checks field has changed for the following products.|'' where id =54';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The National Formulary Indicator has changed for the following|VA Products. The National Formulary Indicator will automatically|be changed in your local DRUG file (#50). Please review the|VISN and Local Formulary designations of these products and|make appropriate changes.'' where id =55';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The following active VA Products are no longer marked for CMOP.|All local drug file entries matched to these VA Products will|be UNMARKED for CMOP. In order to have these entries dispensed|by CMOP any local DRUG file (#50) entries matched to these|products must be re-matched to another VA product that is actively|marked for CMOP dispensing.'' where id =56';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The following VA Products have been added to the National|Drug File.You may wish to review, then match or unmatch|local drug file entries based on this updated information.'' where id =57';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The CS Federal Schedule for the following VA Products has|been added or changed.You may wish to review the DRUG|file (#50) and make appropriate changes to the DEA, SPECIAL|HDLG field (#3).'' where id =58';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The Auto-Create Possible Dosages settings have been edited for the|following VA Products. Please review your local dosages for products|matched to these entries. Edits to your site''''s possible dosages or|local possible dosages may be needed.'' where id =59';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The following VA Print Name(s) have been edited or added. Any product matched|to these products will be unmatched. If site wants to continue to use the|the product the site must rematch to local file #50 entries to the listed VA|product the site must rematch to local file #50 entries to the listed VA|product.'' where id =60';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The following Unit(s) have been edited or added. Any product matched to these|products will be unmatched. If site wants to continue to use the product the|site must rematch to local file #50 entries to the listed VA Product.'' where id =61';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The following VA Dispense Units(s) have been edited or added. Any product|matched to these products will be unmatched. If site wants to continue to use|the product the site must rematch to local file #50 entries to the listed VA|product.'' where id =62';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The following VA Drug Class(es) have been edited or added. The VA Class|for this product will be automatically updated in file #50.'' where id =63';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The following VA Product Identifier(s) have been edited or added. Any product|matched to these products will be unmatched. If site wants to continue to use|the product the site must rematch to local file #50 entries to the listed VA|product.'' where id =64';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The following products will be inactivated on the date listed. No alternative|products have been found.'' where id =65';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The following CS Federal Schedule(s) have been edited or added. Please review|the local DEA special handling field and make edits are required.'' where id =66';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The following VA Generic Name(s) have been edited or added. Any product|matched to these products will be unmatched. If site wants to continue to use|the product the site must rematch to local file #50 entries to the listed VA|product.'' where id =67';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The following VA Dosage Form(s) have been edited. Any product matched to|these products will be unmatched. If site wants to continue to use the|product the site must rematch to local file #50 entries to the listed VA|product.'' where id =68';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The following product(s) have been inactivated. No alternative products have|been found.'' where id =69';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The following VA Product(s) have been reactivated.|'' where id =70';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The FDA Med Guide for the following products has been changed|'' where id =71';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The following products have been flagged for exclusion from drug-drug|interaction checks.'' where id =72';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The following products, previously flagged for exclusion from drug-drug|interaction checks, have been changed to be included|in drug-drug interaction checks where appropriate.'' where id =73';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set STRING_VALUE =''The following interactions in National Drug File (NDF) have been added,|edited or inactivated. These changes are the result of review and|recommendations from the NDF support group.'' where id =74';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set string_value = ''The following VA Print Name(s) have been edited or added. Any product matched|to these products will be unmatched. If site wants to continue to use the|product the site must rematch to local file #50 entries to the listed VA product.'' where id = 60';

      EXECUTE IMMEDIATE
         'UPDATE ppsnepl.epl_national_settings set string_value = ''The following product(s) have been edited Any file|#50 entries matched to these products will be unmatched. To continue to use|the product, the site must rematch the local file #50 entries to the listed VA Product.'' where id = 82';

      COMMIT;

      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   END:  update_national_setting');

      IF vn_row_count = 0
      THEN
         RETURN (1);
      END IF;

      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
               'update_national_setting: Error code '
            || vn_code
            || ': '
            || vc_errm);
         RETURN (1);
   END update_national_setting;

   -----------------------------------------------------------------------------
   -- NAME:         turn_vista_messaging_off
   -- TYPE:         Function
   -- DESCRIPTSION: Update epl_national_setting table to turn vist_messaging_off
   -- INPUTS:       None
   -- OUTPUTS:      Number: (0 - good: >0 - error)
   -----------------------------------------------------------------------------
   FUNCTION turn_vista_messaging_off
      RETURN NUMBER
   IS
   BEGIN
      sys.DBMS_OUTPUT.put_line (CHR (10));
      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   START:  turn_vista_messaging_off');

      EXECUTE IMMEDIATE
         'update ppsnepl.epl_national_settings set boolean_value = 0 where id = 14';

      EXECUTE IMMEDIATE
         'update ppsnepl.epl_national_settings set string_value = null where id = 15';

      EXECUTE IMMEDIATE
         'update ppsnepl.epl_national_settings set integer_value = 0 where id = 16';

      COMMIT;

      sys.DBMS_OUTPUT.put_line (
         'INFORMATION -   END:  turn_vista_messaging_off');

      IF vn_row_count = 0
      THEN
         RETURN (1);
      END IF;

      RETURN (0);
   EXCEPTION
      WHEN OTHERS
      THEN
         vn_code := SQLCODE;
         vc_errm := SUBSTR (SQLERRM, 1, 64);
         sys.DBMS_OUTPUT.put_line (
               'turn_vista_messaging_off: Error code '
            || vn_code
            || ': '
            || vc_errm);
         RETURN (1);
   END turn_vista_messaging_off;
-----------------------------------------------------------------------------
-- MAIN SECTION
-----------------------------------------------------------------------------
BEGIN
   sys.DBMS_OUTPUT.put_line (CHR (10));
   sys.DBMS_OUTPUT.put_line (
      'INFORMATION -   Checking to see if PPS-N 2.0 table(s) already exists.');

   vn_continue := check_for_ppsn2;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -     ONE OR MORE PPS-N 2.0 OBJECTS ALREADY EXISTS. STOPPING MIGRATION.');
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPS-N 2.0 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;

   vn_continue := create_ppsn_tables;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPS-N 2.0 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;

   vn_continue := create_ppsn_views;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPS-N 2.0 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;

   vn_continue := insert_ppsn2_data;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPS-N 2.0 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;

   vn_continue := ndc_process_1;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPS-N 2.0 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;

/* Per Jean Chen, removed and replaced by new process to load from VistA

   vn_continue := ndc_process_2;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPS-N 2.0 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;
*/

   vn_continue := ndc_process_3;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPS-N 2.0 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;

   vn_continue := alter_ppsn2_tables;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPS-N 2.0 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;

   vn_continue := insert_epl_national_settings;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPS-N 2.0 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;

   vn_continue := insert_match_type;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPS-N 2.0 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;

   vn_continue := update_national_setting;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPS-N 2.0 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;

   vn_continue := turn_vista_messaging_off;

   IF vn_continue > 0
   THEN
      sys.DBMS_OUTPUT.put_line (
         'ERROR   -   EXECUTE PPS-N 2.0 ROLLBACK PROCESS');
      sys.DBMS_OUTPUT.put_line ('ERROR   -   EXITING SCRIPT...');
      RETURN;
   END IF;

   -- Recompile Schema and Gather Statistics
   DBMS_UTILITY.compile_schema (schema => 'PPSNEPL');
   DBMS_STATS.gather_schema_stats ('PPSNEPL');
EXCEPTION
   WHEN OTHERS
   THEN
      vn_code := SQLCODE;
      vc_errm := SUBSTR (SQLERRM, 1, 64);
      sys.DBMS_OUTPUT.put_line (
         'MAIN: Error code ' || vn_code || ': ' || vc_errm);
END;
/

SELECT 'Migration Stopped: ' || CURRENT_TIMESTAMP FROM DUAL;

SPOOL OFF
EXIT